import { COOKIE_NAME } from "@shared/const";
import { getSessionCookieOptions } from "./_core/cookies";
import { systemRouter } from "./_core/systemRouter";
import { publicProcedure, protectedProcedure, router } from "./_core/trpc";
import { z } from "zod";
import { createJobPin, getAllJobPins, getJobPinById, updateJobPinStatus, deleteJobPin } from "./db";
import { fetchCottonPinsJobs } from "./fetchCottonPins";
import { storagePut } from "./storage";
import { generateAltText } from "./altTextGenerator";
import { makeRequest, PlaceDetailsResult } from "./_core/map";

// Helper function to convert Unix timestamp to relative time
function getRelativeTime(unixTime: number): string {
  const now = Date.now();
  const reviewDate = unixTime * 1000; // Convert to milliseconds
  const diffMs = now - reviewDate;
  const diffDays = Math.floor(diffMs / (1000 * 60 * 60 * 24));
  const diffMonths = Math.floor(diffDays / 30);
  const diffYears = Math.floor(diffDays / 365);
  
  if (diffDays < 1) return "today";
  if (diffDays === 1) return "yesterday";
  if (diffDays < 7) return `${diffDays} days ago`;
  if (diffDays < 30) return `${Math.floor(diffDays / 7)} weeks ago`;
  if (diffMonths < 12) return `${diffMonths} months ago`;
  return `${diffYears} years ago`;
}

export const appRouter = router({
    // if you need to use socket.io, read and register route in server/_core/index.ts, all api should start with '/api/' so that the gateway can route correctly
  system: systemRouter,
  
  // Google Reviews API
  reviews: router({
    // Get Google reviews for the business
    getGoogleReviews: publicProcedure.query(async () => {
      try {
        const PLACE_ID = "ChIJ5Yx_nqe3-QERAA4BiobW-8M"; // Cotton City Plumbing Place ID (5707 Ave Q S Dr, Lubbock, TX)
        
        const result = await makeRequest<PlaceDetailsResult>(
          "/maps/api/place/details/json",
          {
            place_id: PLACE_ID,
            fields: "name,rating,user_ratings_total,reviews"
          }
        );
        
        if (result.status !== "OK" || !result.result.reviews) {
          return {
            reviews: [],
            rating: 0,
            totalReviews: 0
          };
        }
        
        // Sort by most recent and return top 5
        const sortedReviews = result.result.reviews
          .sort((a, b) => b.time - a.time)
          .slice(0, 5)
          .map(review => ({
            authorName: review.author_name,
            rating: review.rating,
            text: review.text,
            time: review.time,
            relativeTime: getRelativeTime(review.time)
          }));
        
        return {
          reviews: sortedReviews,
          rating: result.result.rating || 0,
          totalReviews: result.result.user_ratings_total || 0
        };
      } catch (error) {
        console.error("Error fetching Google reviews:", error);
        return {
          reviews: [],
          rating: 0,
          totalReviews: 0
        };
      }
    })
  }),
  auth: router({
    me: publicProcedure.query(opts => opts.ctx.user),
    logout: publicProcedure.mutation(({ ctx }) => {
      const cookieOptions = getSessionCookieOptions(ctx.req);
      ctx.res.clearCookie(COOKIE_NAME, { ...cookieOptions, maxAge: -1 });
      return {
        success: true,
      } as const;
    }),
  }),

  // Job Pins API for Data Pins integration
  jobPins: router({
    // Get all job pins (public - for displaying on the website)
    // Fetches from Cotton Pins website in real-time
    list: publicProcedure.query(async () => {
      try {
        // Fetch from Cotton Pins website
        const cottonPins = await fetchCottonPinsJobs();
        
        if (cottonPins.length > 0) {
          // Transform Cotton Pins data to match our JobPin schema
          return cottonPins.map((pin, index) => ({
            id: index + 1, // Temporary ID for display
            title: pin.title,
            description: pin.description,
            serviceType: pin.serviceType,
            address: pin.address,
            latitude: pin.latitude,
            longitude: pin.longitude,
            photoUrl: pin.photoUrl,
            photoKey: `cotton-pins/${pin.id}`,
            technicianName: null,
            customerReview: null,
            customerName: null,
            rating: null,
            altText: `${pin.title} - ${pin.description}`,
            createdBy: 1,
            status: "approved" as const,
            createdAt: new Date(pin.date || Date.now()),
            updatedAt: new Date(pin.date || Date.now()),
          }));
        }
        
        // Fallback to local database if Cotton Pins fetch fails
        const pins = await getAllJobPins();
        return pins;
      } catch (error) {
        console.error('[JobPins] Failed to fetch from Cotton Pins:', error);
        // Fallback to local database
        const pins = await getAllJobPins();
        return pins;
      }
    }),

    // Get a single job pin by ID (public)
    getById: publicProcedure
      .input(z.object({ id: z.number() }))
      .query(async ({ input }) => {
        try {
          // First, try to fetch from Cotton Pins API
          const cottonPins = await fetchCottonPinsJobs();
          
          if (cottonPins.length > 0) {
            // Find the job by index (since we use index + 1 as temporary ID)
            const jobIndex = input.id - 1;
            if (jobIndex >= 0 && jobIndex < cottonPins.length) {
              const pin = cottonPins[jobIndex];
              return {
                id: input.id,
                title: pin.title,
                description: pin.description,
                serviceType: pin.serviceType,
                address: pin.address,
                latitude: pin.latitude,
                longitude: pin.longitude,
                photoUrl: pin.photoUrl,
                photoKey: `cotton-pins/${pin.id}`,
                technicianName: null,
                customerReview: null,
                customerName: null,
                rating: null,
                altText: `${pin.title} - ${pin.description}`,
                createdBy: 1,
                status: "approved" as const,
                createdAt: new Date(pin.date || Date.now()),
                updatedAt: new Date(pin.date || Date.now()),
              };
            }
          }
          
          // Fallback to local database
          const pin = await getJobPinById(input.id);
          if (!pin) {
            throw new Error(`Job pin with ID ${input.id} not found`);
          }
          return pin;
        } catch (error) {
          console.error('[JobPins] Failed to fetch job pin:', error);
          // Try local database as last resort
          const pin = await getJobPinById(input.id);
          if (!pin) {
            throw new Error(`Job pin with ID ${input.id} not found`);
          }
          return pin;
        }
      }),

    // Create a new job pin (protected - requires authentication)
    create: protectedProcedure
      .input(
        z.object({
          title: z.string().min(1).max(255),
          description: z.string().min(1),
          serviceType: z.string().min(1).max(100),
          address: z.string().min(1).max(500),
          latitude: z.string().min(1).max(50),
          longitude: z.string().min(1).max(50),
          photoBase64: z.string(), // Base64 encoded photo
          technicianName: z.string().optional(),
          customerReview: z.string().optional(),
          customerName: z.string().optional(),
          rating: z.number().min(1).max(5).optional(),
        })
      )
      .mutation(async ({ input, ctx }) => {
        // Upload photo to S3
        const photoBuffer = Buffer.from(input.photoBase64, "base64");
        const timestamp = Date.now();
        const photoKey = `job-pins/${ctx.user.id}-${timestamp}.jpg`;
        const { url: photoUrl } = await storagePut(photoKey, photoBuffer, "image/jpeg");

        // Generate AI alt text for the photo
        const altText = await generateAltText(
          photoUrl,
          input.title,
          input.serviceType,
          input.address
        );

        // Create job pin in database with auto-approved status
        const result = await createJobPin({
          title: input.title,
          description: input.description,
          serviceType: input.serviceType,
          address: input.address,
          latitude: input.latitude,
          longitude: input.longitude,
          photoUrl,
          photoKey,
          technicianName: input.technicianName || null,
          customerReview: input.customerReview || null,
          customerName: input.customerName || null,
          rating: input.rating || null,
          altText: altText,
          createdBy: ctx.user.id,
          status: "approved", // Auto-approve all new pins
        });

        return { success: true, id: result[0].insertId };
      }),

    // Admin: Get all job pins including pending/hidden (requires admin role)
    listAll: protectedProcedure.query(async ({ ctx }) => {
      if (ctx.user.role !== "admin") {
        throw new Error("Admin access required");
      }
      const pins = await getAllJobPins(true); // Include all statuses
      return pins;
    }),

    // Admin: Update job pin status
    updateStatus: protectedProcedure
      .input(
        z.object({
          id: z.number(),
          status: z.enum(["pending", "approved", "featured", "hidden"]),
        })
      )
      .mutation(async ({ input, ctx }) => {
        if (ctx.user.role !== "admin") {
          throw new Error("Admin access required");
        }
        await updateJobPinStatus(input.id, input.status);
        return { success: true };
      }),

    // Admin: Delete job pin
    delete: protectedProcedure
      .input(z.object({ id: z.number() }))
      .mutation(async ({ input, ctx }) => {
        if (ctx.user.role !== "admin") {
          throw new Error("Admin access required");
        }
        await deleteJobPin(input.id);
        return { success: true };
      }),
  }),
});

export type AppRouter = typeof appRouter;
