import { describe, expect, it } from "vitest";
import { appRouter } from "./routers";
import type { TrpcContext } from "./_core/context";

type AuthenticatedUser = NonNullable<TrpcContext["user"]>;

function createAuthContext(): { ctx: TrpcContext } {
  const user: AuthenticatedUser = {
    id: 1,
    openId: "sample-user",
    email: "sample@example.com",
    name: "Sample User",
    loginMethod: "manus",
    role: "user",
    createdAt: new Date(),
    updatedAt: new Date(),
    lastSignedIn: new Date(),
  };

  const ctx: TrpcContext = {
    user,
    req: {
      protocol: "https",
      headers: {},
    } as TrpcContext["req"],
    res: {
      clearCookie: () => {},
    } as TrpcContext["res"],
  };

  return { ctx };
}

describe("jobPins.list", () => {
  it("returns an array of job pins", async () => {
    const { ctx } = createAuthContext();
    const caller = appRouter.createCaller(ctx);

    const result = await caller.jobPins.list();

    expect(Array.isArray(result)).toBe(true);
  });
});

describe("jobPins.getById", () => {
  it("returns undefined for non-existent job pin", async () => {
    const { ctx } = createAuthContext();
    const caller = appRouter.createCaller(ctx);

    const result = await caller.jobPins.getById({ id: 999999 });

    expect(result).toBeUndefined();
  });
});

describe("jobPins.create", () => {
  it("requires authentication", async () => {
    const ctx: TrpcContext = {
      user: null,
      req: {
        protocol: "https",
        headers: {},
      } as TrpcContext["req"],
      res: {
        clearCookie: () => {},
      } as TrpcContext["res"],
    };

    const caller = appRouter.createCaller(ctx);

    await expect(
      caller.jobPins.create({
        title: "Test Job",
        description: "Test description",
        serviceType: "Water Heaters",
        address: "123 Test St, Lubbock, TX",
        latitude: "33.5779",
        longitude: "-101.8552",
        photoBase64: "iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mNk+M9QDwADhgGAWjR9awAAAABJRU5ErkJggg==",
      })
    ).rejects.toThrow();
  });

  it("creates a job pin successfully when authenticated", async () => {
    const { ctx } = createAuthContext();
    const caller = appRouter.createCaller(ctx);

    const result = await caller.jobPins.create({
      title: "Test Water Heater Installation",
      description: "Installed a new tankless water heater for a customer in Lubbock.",
      serviceType: "Water Heaters",
      address: "123 Test St, Lubbock, TX 79401",
      latitude: "33.5779",
      longitude: "-101.8552",
      photoBase64: "iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mNk+M9QDwADhgGAWjR9awAAAABJRU5ErkJggg==",
      technicianName: "Test Technician",
      customerReview: "Great service!",
      customerName: "Test Customer",
      rating: 5,
    });

    expect(result.success).toBe(true);
    expect(result.id).toBeDefined();
  });
});
