import { describe, expect, it } from "vitest";
import { appRouter } from "./routers";
import type { TrpcContext } from "./_core/context";

type AuthenticatedUser = NonNullable<TrpcContext["user"]>;

function createAdminContext(): { ctx: TrpcContext } {
  const user: AuthenticatedUser = {
    id: 1,
    openId: "admin-user",
    email: "admin@cottoncityplumbing.com",
    name: "Admin User",
    loginMethod: "manus",
    role: "admin",
    createdAt: new Date(),
    updatedAt: new Date(),
    lastSignedIn: new Date(),
  };

  const ctx: TrpcContext = {
    user,
    req: {
      protocol: "https",
      headers: {},
    } as TrpcContext["req"],
    res: {
      clearCookie: () => {},
    } as TrpcContext["res"],
  };

  return { ctx };
}

function createRegularUserContext(): { ctx: TrpcContext } {
  const user: AuthenticatedUser = {
    id: 2,
    openId: "regular-user",
    email: "user@example.com",
    name: "Regular User",
    loginMethod: "manus",
    role: "user",
    createdAt: new Date(),
    updatedAt: new Date(),
    lastSignedIn: new Date(),
  };

  const ctx: TrpcContext = {
    user,
    req: {
      protocol: "https",
      headers: {},
    } as TrpcContext["req"],
    res: {
      clearCookie: () => {},
    } as TrpcContext["res"],
  };

  return { ctx };
}

describe("Admin Job Pins", () => {
  describe("jobPins.listAll", () => {
    it("allows admin to view all job pins", async () => {
      const { ctx } = createAdminContext();
      const caller = appRouter.createCaller(ctx);

      const result = await caller.jobPins.listAll();

      expect(Array.isArray(result)).toBe(true);
    });

    it("denies regular users from viewing all job pins", async () => {
      const { ctx } = createRegularUserContext();
      const caller = appRouter.createCaller(ctx);

      await expect(caller.jobPins.listAll()).rejects.toThrow("Admin access required");
    });
  });

  describe("jobPins.updateStatus", () => {
    it("allows admin to update job pin status", async () => {
      const { ctx } = createAdminContext();
      const caller = appRouter.createCaller(ctx);

      // First create a job pin
      const createResult = await caller.jobPins.create({
        title: "Test Job for Status Update",
        description: "This job will have its status updated.",
        serviceType: "Water Heaters",
        address: "123 Test St, Lubbock, TX 79401",
        latitude: "33.5779",
        longitude: "-101.8552",
        photoBase64: "iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mNk+M9QDwADhgGAWjR9awAAAABJRU5ErkJggg==",
      });

      const jobId = createResult.id;

      // Update status to approved
      const updateResult = await caller.jobPins.updateStatus({
        id: jobId,
        status: "approved",
      });

      expect(updateResult.success).toBe(true);

      // Verify the status was updated
      const job = await caller.jobPins.getById({ id: jobId });
      expect(job?.status).toBe("approved");
    });

    it("denies regular users from updating status", async () => {
      const { ctx } = createRegularUserContext();
      const caller = appRouter.createCaller(ctx);

      await expect(
        caller.jobPins.updateStatus({ id: 1, status: "approved" })
      ).rejects.toThrow("Admin access required");
    });
  });

  describe("jobPins.delete", () => {
    it("allows admin to delete job pins", async () => {
      const { ctx } = createAdminContext();
      const caller = appRouter.createCaller(ctx);

      // First create a job pin
      const createResult = await caller.jobPins.create({
        title: "Test Job for Deletion",
        description: "This job will be deleted.",
        serviceType: "Drain Cleaning",
        address: "456 Test Ave, Lubbock, TX 79401",
        latitude: "33.5779",
        longitude: "-101.8552",
        photoBase64: "iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mNk+M9QDwADhgGAWjR9awAAAABJRU5ErkJggg==",
      });

      const jobId = createResult.id;

      // Delete the job pin
      const deleteResult = await caller.jobPins.delete({ id: jobId });
      expect(deleteResult.success).toBe(true);

      // Verify it's deleted
      const job = await caller.jobPins.getById({ id: jobId });
      expect(job).toBeUndefined();
    });

    it("denies regular users from deleting job pins", async () => {
      const { ctx } = createRegularUserContext();
      const caller = appRouter.createCaller(ctx);

      await expect(caller.jobPins.delete({ id: 1 })).rejects.toThrow("Admin access required");
    });
  });

  describe("Public visibility filtering", () => {
    it("public list only shows approved and featured pins", async () => {
      const { ctx } = createAdminContext();
      const caller = appRouter.createCaller(ctx);

      // Create a pending job pin
      const pendingJob = await caller.jobPins.create({
        title: "Pending Job",
        description: "This job is pending approval.",
        serviceType: "Water Heaters",
        address: "789 Pending St, Lubbock, TX 79401",
        latitude: "33.5779",
        longitude: "-101.8552",
        photoBase64: "iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAADUlEQVR42mNk+M9QDwADhgGAWjR9awAAAABJRU5ErkJggg==",
      });

      // Public list should not include pending jobs
      const publicList = await caller.jobPins.list();
      const pendingInPublicList = publicList.find(job => job.id === pendingJob.id);
      expect(pendingInPublicList).toBeUndefined();

      // Admin list should include pending jobs
      const adminList = await caller.jobPins.listAll();
      const pendingInAdminList = adminList.find(job => job.id === pendingJob.id);
      expect(pendingInAdminList).toBeDefined();
      expect(pendingInAdminList?.status).toBe("pending");
    });
  });
});
