/**
 * Simple script to import Cotton Pins jobs from JSON file to database
 * Run with: node import-cotton-pins.mjs
 */

import { readFileSync } from 'fs';
import { drizzle } from 'drizzle-orm/mysql2';
import { jobPins } from './drizzle/schema.js';

// Read the JSON file
const jsonData = JSON.parse(readFileSync('/home/ubuntu/data_pins_jobs_complete.json', 'utf8'));

console.log(`Found ${jsonData.length} jobs to import`);

// Connect to database
const db = drizzle(process.env.DATABASE_URL);

// Import each job
let imported = 0;
let skipped = 0;

for (const job of jsonData) {
  try {
    // Extract address from description or use location
    const address = job.location === "Lubbock" ? "Lubbock, TX" : `${job.location}, TX`;
    
    await db.insert(jobPins).values({
      title: job.title,
      description: job.description,
      serviceType: job.serviceType,
      address: address,
      latitude: "33.5779", // Default Lubbock coordinates
      longitude: "-101.8552",
      photoUrl: job.imageUrl,
      photoKey: `cotton-pins-import/${Date.now()}`,
      technicianName: null,
      customerReview: null,
      customerName: null,
      rating: null,
      altText: `${job.title} - ${job.description}`,
      createdBy: 1,
      status: "approved",
    });
    
    imported++;
    console.log(`✓ Imported: ${job.title}`);
  } catch (error) {
    if (error.message && error.message.includes('Duplicate')) {
      skipped++;
      console.log(`- Skipped (duplicate): ${job.title}`);
    } else {
      console.error(`✗ Failed to import "${job.title}":`, error.message);
    }
  }
}

console.log(`\nImport complete!`);
console.log(`Imported: ${imported}`);
console.log(`Skipped: ${skipped}`);
console.log(`Total: ${jsonData.length}`);

process.exit(0);
