import Layout from "@/components/Layout";
import SEO from "@/components/SEO";
import { Button } from "@/components/ui/button";
import { Card, CardContent } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { MapPin, Calendar, User, Star, Share2, Facebook, Twitter, Linkedin, Phone, Mail, ArrowLeft } from "lucide-react";
import { Link, useRoute } from "wouter";
import { MapView } from "@/components/Map";
import { useState } from "react";
import { trpc } from "@/lib/trpc";
import { JobPinSchema } from "@/components/JobPinSchema";
import { Helmet } from "react-helmet-async";

export default function JobPinDetail() {
  const [, params] = useRoute("/recent-jobs/:id");
  const jobId = params?.id ? parseInt(params.id) : 0;
  
  const [mapInstance, setMapInstance] = useState<google.maps.Map | null>(null);
  const [shareMenuOpen, setShareMenuOpen] = useState(false);

  // Fetch the specific job pin
  const { data: jobPin, isLoading, error } = trpc.jobPins.getById.useQuery({ id: jobId });

  // Fetch all job pins for "related jobs"
  const { data: allJobPins } = trpc.jobPins.list.useQuery();

  // Get related jobs (same service type or nearby location)
  const relatedJobs = allJobPins?.filter(job => 
    job.id !== jobId && 
    (job.serviceType === jobPin?.serviceType || 
     job.address.includes(jobPin?.address.split(',')[1]?.trim() || ''))
  ).slice(0, 3);

  const onMapReady = (map: google.maps.Map) => {
    setMapInstance(map);
    
    if (jobPin) {
      const marker = new google.maps.Marker({
        position: { lat: parseFloat(jobPin.latitude), lng: parseFloat(jobPin.longitude) },
        map: map,
        title: jobPin.title,
        animation: google.maps.Animation.DROP,
      });

      const infoWindow = new google.maps.InfoWindow({
        content: `
          <div style="padding: 8px;">
            <h3 style="font-weight: bold; margin-bottom: 4px;">${jobPin.title}</h3>
            <p style="font-size: 12px; color: #666;">${jobPin.address}</p>
          </div>
        `,
      });

      marker.addListener("click", () => {
        infoWindow.open(map, marker);
      });
    }
  };

  const shareUrl = typeof window !== 'undefined' ? window.location.href : '';
  const shareTitle = jobPin ? `${jobPin.title} - Cotton City Plumbing` : '';
  const shareDescription = jobPin?.description || '';

  const handleShare = (platform: string) => {
    const encodedUrl = encodeURIComponent(shareUrl);
    const encodedTitle = encodeURIComponent(shareTitle);
    const encodedDescription = encodeURIComponent(shareDescription);

    const urls: Record<string, string> = {
      facebook: `https://www.facebook.com/sharer/sharer.php?u=${encodedUrl}`,
      twitter: `https://twitter.com/intent/tweet?url=${encodedUrl}&text=${encodedTitle}`,
      linkedin: `https://www.linkedin.com/sharing/share-offsite/?url=${encodedUrl}`,
    };

    if (urls[platform]) {
      window.open(urls[platform], '_blank', 'width=600,height=400');
    }
  };

  if (isLoading) {
    return (
      <Layout>
        <div className="min-h-screen flex items-center justify-center">
          <div className="text-center">
            <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-red-600 mx-auto mb-4"></div>
            <p className="text-gray-600">Loading job details...</p>
          </div>
        </div>
      </Layout>
    );
  }

  if (error || !jobPin) {
    return (
      <Layout>
        <SEO 
          title="Job Not Found | Cotton City Plumbing"
          description="The requested job could not be found."
          canonical="/recent-jobs"
        />
        <div className="min-h-screen flex items-center justify-center">
          <div className="text-center">
            <h1 className="text-4xl font-bold text-zinc-900 mb-4">Job Not Found</h1>
            <p className="text-gray-600 mb-8">The job you're looking for doesn't exist or has been removed.</p>
            <Link href="/recent-jobs">
              <Button className="bg-red-600 hover:bg-red-700">
                <ArrowLeft className="h-4 w-4 mr-2" />
                Back to Recent Jobs
              </Button>
            </Link>
          </div>
        </div>
      </Layout>
    );
  }

  const city = jobPin.address.split(',')[1]?.trim() || 'Lubbock';
  const pageTitle = `${jobPin.title} in ${city} | Cotton City Plumbing`;
  const pageDescription = `${jobPin.description.substring(0, 150)}... Professional ${jobPin.serviceType} service in ${city}, TX by Cotton City Plumbing.`;

  return (
    <Layout>
      <SEO 
        title={pageTitle}
        description={pageDescription}
        canonical={`/recent-jobs/${jobPin.id}`}
      />
      
      <JobPinSchema jobPin={jobPin} />

      {/* Open Graph tags for social sharing */}
      <Helmet>
        <meta property="og:title" content={pageTitle} />
        <meta property="og:description" content={pageDescription} />
        <meta property="og:image" content={jobPin.photoUrl} />
        <meta property="og:url" content={shareUrl} />
        <meta property="og:type" content="article" />
        <meta name="twitter:card" content="summary_large_image" />
        <meta name="twitter:title" content={pageTitle} />
        <meta name="twitter:description" content={pageDescription} />
        <meta name="twitter:image" content={jobPin.photoUrl} />
      </Helmet>

      {/* Breadcrumb Navigation */}
      <section className="bg-gray-100 py-4">
        <div className="container">
          <nav className="flex items-center gap-2 text-sm text-gray-600">
            <Link href="/" className="hover:text-red-600">Home</Link>
            <span>/</span>
            <Link href="/recent-jobs" className="hover:text-red-600">Recent Jobs</Link>
            <span>/</span>
            <span className="text-zinc-900 font-bold">{jobPin.title}</span>
          </nav>
        </div>
      </section>

      {/* Hero Section */}
      <section className="bg-black text-white py-12">
        <div className="container">
          <Link href="/recent-jobs">
            <Button variant="outline" className="mb-4 border-white text-white hover:bg-white hover:text-black">
              <ArrowLeft className="h-4 w-4 mr-2" />
              Back to Recent Jobs
            </Button>
          </Link>
          <div className="flex flex-wrap items-start justify-between gap-4">
            <div className="flex-1">
              <Badge className="mb-3 bg-red-600 text-white">{jobPin.serviceType}</Badge>
              <h1 className="font-league-gothic text-4xl md:text-6xl mb-4 uppercase tracking-wide">{jobPin.title}</h1>
              <div className="flex flex-wrap items-center gap-4 text-gray-300">
                <div className="flex items-center gap-2">
                  <MapPin className="h-5 w-5" />
                  <span>{jobPin.address}</span>
                </div>
                <div className="flex items-center gap-2">
                  <Calendar className="h-5 w-5" />
                  <span>{new Date(jobPin.createdAt).toLocaleDateString('en-US', { 
                    month: 'long', 
                    day: 'numeric', 
                    year: 'numeric' 
                  })}</span>
                </div>
                {jobPin.technicianName && (
                  <div className="flex items-center gap-2">
                    <User className="h-5 w-5" />
                    <span>{jobPin.technicianName}</span>
                  </div>
                )}
              </div>
            </div>
            <div className="relative">
              <Button
                onClick={() => setShareMenuOpen(!shareMenuOpen)}
                variant="outline"
                className="border-white text-white hover:bg-white hover:text-black"
              >
                <Share2 className="h-4 w-4 mr-2" />
                Share
              </Button>
              {shareMenuOpen && (
                <div className="absolute right-0 mt-2 bg-white rounded-lg shadow-xl p-2 z-10 flex gap-2">
                  <Button
                    size="sm"
                    variant="outline"
                    onClick={() => handleShare('facebook')}
                    className="text-blue-600 hover:bg-blue-50"
                  >
                    <Facebook className="h-4 w-4" />
                  </Button>
                  <Button
                    size="sm"
                    variant="outline"
                    onClick={() => handleShare('twitter')}
                    className="text-sky-500 hover:bg-sky-50"
                  >
                    <Twitter className="h-4 w-4" />
                  </Button>
                  <Button
                    size="sm"
                    variant="outline"
                    onClick={() => handleShare('linkedin')}
                    className="text-blue-700 hover:bg-blue-50"
                  >
                    <Linkedin className="h-4 w-4" />
                  </Button>
                </div>
              )}
            </div>
          </div>
        </div>
      </section>

      {/* Main Content */}
      <section className="py-12 bg-white">
        <div className="container">
          <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
            {/* Left Column - Photo and Description */}
            <div className="lg:col-span-2 space-y-8">
              {/* Photo */}
              <div className="rounded-xl overflow-hidden shadow-2xl">
                <img 
                  src={jobPin.photoUrl} 
                  alt={jobPin.altText || jobPin.title} 
                  className="w-full h-auto object-cover"
                />
              </div>

              {/* Description */}
              <Card>
                <CardContent className="p-6">
                  <h2 className="font-league-gothic text-3xl text-zinc-900 mb-4 uppercase">Job Details</h2>
                  <p className="text-gray-700 text-lg leading-relaxed whitespace-pre-line">
                    {jobPin.description}
                  </p>
                </CardContent>
              </Card>

              {/* Customer Review */}
              {jobPin.customerReview && (
                <Card className="border-l-4 border-l-yellow-400">
                  <CardContent className="p-6">
                    <h2 className="font-league-gothic text-3xl text-zinc-900 mb-4 uppercase">Customer Review</h2>
                    {jobPin.rating && (
                      <div className="flex items-center gap-1 mb-3">
                        {[...Array(5)].map((_, i) => (
                          <Star 
                            key={i} 
                            className={`h-6 w-6 ${i < jobPin.rating! ? 'fill-yellow-400 text-yellow-400' : 'text-gray-300'}`}
                          />
                        ))}
                        <span className="ml-2 text-2xl font-bold text-zinc-900">{jobPin.rating}.0</span>
                      </div>
                    )}
                    <blockquote className="text-gray-700 text-lg italic leading-relaxed mb-3">
                      "{jobPin.customerReview}"
                    </blockquote>
                    {jobPin.customerName && (
                      <p className="text-zinc-900 font-bold">— {jobPin.customerName}</p>
                    )}
                  </CardContent>
                </Card>
              )}

              {/* Map */}
              <Card>
                <CardContent className="p-6">
                  <h2 className="font-league-gothic text-3xl text-zinc-900 mb-4 uppercase">Job Location</h2>
                  <div className="h-[400px] rounded-lg overflow-hidden">
                    <MapView 
                      className="w-full h-full"
                      initialCenter={{ lat: parseFloat(jobPin.latitude), lng: parseFloat(jobPin.longitude) }}
                      initialZoom={15}
                      onMapReady={onMapReady}
                    />
                  </div>
                  <p className="text-gray-600 mt-3 flex items-center gap-2">
                    <MapPin className="h-4 w-4" />
                    {jobPin.address}
                  </p>
                </CardContent>
              </Card>
            </div>

            {/* Right Column - CTA and Related Jobs */}
            <div className="space-y-6">
              {/* Call to Action */}
              <Card className="bg-red-600 text-white sticky top-24">
                <CardContent className="p-6 text-center">
                  <h3 className="font-league-gothic text-3xl mb-3 uppercase">Need This Service?</h3>
                  <p className="mb-6">Get the same professional service in your area!</p>
                  <div className="space-y-3">
                    <a href="tel:8064414110" className="block">
                      <Button size="lg" className="w-full bg-white text-red-600 hover:bg-gray-100 font-bold">
                        <Phone className="h-5 w-5 mr-2" />
                        Call (806) 441-4110
                      </Button>
                    </a>
                    <Link href="/contact" className="block">
                      <Button size="lg" variant="outline" className="w-full border-2 border-white text-white hover:bg-white hover:text-red-600 font-bold">
                        <Mail className="h-5 w-5 mr-2" />
                        Schedule Online
                      </Button>
                    </Link>
                  </div>
                </CardContent>
              </Card>

              {/* Related Jobs */}
              {relatedJobs && relatedJobs.length > 0 && (
                <Card>
                  <CardContent className="p-6">
                    <h3 className="font-league-gothic text-2xl text-zinc-900 mb-4 uppercase">Related Jobs</h3>
                    <div className="space-y-4">
                      {relatedJobs.map((job) => (
                        <Link key={job.id} href={`/recent-jobs/${job.id}`}>
                          <div className="group cursor-pointer border border-gray-200 rounded-lg p-3 hover:border-red-600 hover:shadow-md transition-all">
                            <div className="flex gap-3">
                    <img 
                      src={job.photoUrl} 
                      alt={job.altText || job.title}
                                className="w-20 h-20 object-cover rounded"
                              />
                              <div className="flex-1 min-w-0">
                                <h4 className="font-bold text-sm text-zinc-900 group-hover:text-red-600 line-clamp-2 mb-1">
                                  {job.title}
                                </h4>
                                <p className="text-xs text-gray-500 flex items-center gap-1">
                                  <MapPin className="h-3 w-3" />
                                  {job.address.split(',').slice(-2).join(',').trim()}
                                </p>
                              </div>
                            </div>
                          </div>
                        </Link>
                      ))}
                    </div>
                    <Link href="/recent-jobs">
                      <Button variant="outline" className="w-full mt-4">
                        View All Recent Jobs
                      </Button>
                    </Link>
                  </CardContent>
                </Card>
              )}
            </div>
          </div>
        </div>
      </section>
    </Layout>
  );
}
