import Layout from "@/components/Layout";
import SEO from "@/components/SEO";
import { Button } from "@/components/ui/button";
import { Card, CardContent } from "@/components/ui/card";
import { Check, Phone, MapPin } from "lucide-react";
import { Link } from "wouter";

interface ServiceAreaProps {
  city: string;
  introText: string;
  mapEmbedUrl?: string;
}

export default function ServiceAreaLayout({
  city,
  introText,
  mapEmbedUrl
}: ServiceAreaProps) {
  const title = `Plumber in ${city}, TX`;
  const description = `Expert plumbing services in ${city}, TX. Cotton City Plumbing offers water heater repair, drain cleaning, and emergency plumbing for ${city} residents. Call (806) 441-4110.`;

  return (
    <Layout>
      <SEO 
        title={title}
        description={description}
        keywords={`plumber ${city}, plumbing services ${city} tx, water heater repair ${city}, drain cleaning ${city}`}
        canonical={`/areas/${city.toLowerCase()}`}
      />
      
      {/* Hero Section */}
      <section className="relative h-[40vh] min-h-[300px] flex items-center justify-center overflow-hidden bg-zinc-900">
        <div className="absolute inset-0 z-0">
          <img 
            src="/images/hero-truck.png" 
            alt={`Plumber in ${city}`} 
            className="w-full h-full object-cover opacity-30"
          />
          <div className="absolute inset-0 bg-gradient-to-t from-zinc-900 via-transparent to-transparent"></div>
        </div>
        <div className="container relative z-10 text-white text-center">
          <h1 className="font-league-gothic text-5xl md:text-7xl mb-4 uppercase tracking-wide">
            Plumber in <span className="text-red-600">{city}, TX</span>
          </h1>
          <p className="text-xl md:text-2xl text-gray-300 font-light max-w-2xl mx-auto">
            Reliable, Local Plumbing Services for {city} Homeowners
          </p>
        </div>
      </section>

      <div className="container py-16">
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-12">
          {/* Main Content */}
          <div className="lg:col-span-2 space-y-12">
            <section>
              <h2 className="font-league-gothic text-4xl text-zinc-900 mb-6 uppercase">
                Trusted Plumbing Experts Serving {city}
              </h2>
              <p className="text-lg text-gray-600 leading-relaxed mb-8 whitespace-pre-line">
                {introText}
              </p>
              
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-8">
                {[
                  "Licensed & Insured",
                  "Upfront Pricing",
                  "Same-Day Service",
                  "Emergency Repairs",
                  "Local Experts",
                  "Satisfaction Guaranteed"
                ].map((benefit, index) => (
                  <div key={index} className="flex items-center gap-3">
                    <div className="bg-red-100 p-1 rounded-full">
                      <Check className="h-4 w-4 text-red-600" />
                    </div>
                    <span className="font-medium text-gray-800">{benefit}</span>
                  </div>
                ))}
              </div>

              <div className="flex flex-col sm:flex-row gap-4">
                <a href="tel:8064414110">
                  <Button size="lg" className="bg-red-600 hover:bg-red-700 text-white font-bold text-lg px-8 py-6 w-full sm:w-auto uppercase tracking-wider">
                    <Phone className="mr-2 h-5 w-5" /> Call Now
                  </Button>
                </a>
                <Link href="/contact">
                  <Button size="lg" variant="outline" className="border-2 border-zinc-900 text-zinc-900 hover:bg-zinc-900 hover:text-white font-bold text-lg px-8 py-6 w-full sm:w-auto uppercase tracking-wider">
                    Schedule Online
                  </Button>
                </Link>
              </div>
            </section>

            <section>
              <h3 className="font-league-gothic text-3xl text-zinc-900 mb-6 uppercase">
                Common Plumbing Services in {city}
              </h3>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                {[
                  { title: "Water Heater Repair", desc: "Fast repair and replacement for tank and tankless water heaters." },
                  { title: "Drain Cleaning", desc: "Clearing clogged drains and sewer lines to restore proper flow." },
                  { title: "Leak Detection", desc: "Locating and fixing hidden water leaks to prevent property damage." },
                  { title: "Slab Leak Repair", desc: "Expert diagnosis and repair of leaks under your foundation." },
                  { title: "Gas Line Services", desc: "Safe installation and repair of gas lines for appliances and heating." },
                  { title: "Water Treatment", desc: "Installation of water softeners and filtration systems for hard water." }
                ].map((service, index) => (
                  <Card key={index} className="border-l-4 border-l-red-600 shadow-sm hover:shadow-md transition-shadow">
                    <CardContent className="p-6">
                      <h4 className="font-bold text-lg text-zinc-900 mb-2 uppercase tracking-wide">{service.title}</h4>
                      <p className="text-sm text-gray-600">{service.desc}</p>
                    </CardContent>
                  </Card>
                ))}
              </div>
            </section>
          </div>

          {/* Sidebar */}
          <div className="space-y-8">
            <Card className="bg-zinc-900 text-white border-none shadow-xl sticky top-24">
              <CardContent className="p-8 text-center space-y-6">
                <MapPin className="w-12 h-12 text-red-600 mx-auto" />
                <h3 className="font-league-gothic text-3xl uppercase tracking-wide">Serving {city}</h3>
                <p className="text-gray-300">
                  Cotton City Plumbing is proud to be your local plumber in {city}, TX. We are just a phone call away!
                </p>
                <a href="tel:8064414110" className="block">
                  <Button size="lg" className="w-full bg-red-600 text-white hover:bg-red-700 font-bold uppercase tracking-wider">
                    (806) 441-4110
                  </Button>
                </a>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </Layout>
  );
}
