import mysql from 'mysql2/promise';

const DATABASE_URL = process.env.DATABASE_URL;

async function analyzeDuplicates() {
  const connection = await mysql.createConnection(DATABASE_URL);
  
  try {
    const [rows] = await connection.execute(
      'SELECT id, title, description, address, latitude, longitude, photoUrl, createdAt FROM jobPins ORDER BY title, address'
    );
    
    console.log(`\n${'='.repeat(120)}`);
    console.log(`COMPLETE JOB LIST (${rows.length} total jobs)`);
    console.log('='.repeat(120));
    
    rows.forEach((job, index) => {
      console.log(`\n${index + 1}. ID: ${job.id}`);
      console.log(`   Title: ${job.title}`);
      console.log(`   Description: ${job.description?.substring(0, 80)}...`);
      console.log(`   Address: ${job.address}`);
      console.log(`   GPS: ${job.latitude}, ${job.longitude}`);
      console.log(`   Photo: ${job.photoUrl?.substring(0, 60)}...`);
      console.log(`   Created: ${new Date(job.createdAt).toLocaleString()}`);
    });
    
    // Find duplicates by comparing GPS coordinates (within 0.0001 degrees ~ 10 meters)
    console.log(`\n\n${'='.repeat(120)}`);
    console.log('POTENTIAL DUPLICATES (by GPS proximity)');
    console.log('='.repeat(120));
    
    const duplicateGroups = [];
    const processed = new Set();
    
    rows.forEach((job1, i) => {
      if (processed.has(job1.id)) return;
      
      const group = [job1];
      
      rows.forEach((job2, j) => {
        if (i !== j && !processed.has(job2.id)) {
          const latDiff = Math.abs(job1.latitude - job2.latitude);
          const lonDiff = Math.abs(job1.longitude - job2.longitude);
          
          // Same location (within 10 meters)
          if (latDiff < 0.0001 && lonDiff < 0.0001) {
            group.push(job2);
            processed.add(job2.id);
          }
        }
      });
      
      if (group.length > 1) {
        duplicateGroups.push(group);
        group.forEach(j => processed.add(j.id));
      }
    });
    
    if (duplicateGroups.length > 0) {
      duplicateGroups.forEach((group, index) => {
        console.log(`\n--- Duplicate Group ${index + 1} (${group.length} jobs at same location) ---`);
        group.forEach(job => {
          console.log(`  ID: ${job.id} | ${job.title} | ${job.address}`);
          console.log(`    GPS: ${job.latitude}, ${job.longitude}`);
          console.log(`    Photo: ${job.photoUrl?.split('/').pop()}`);
        });
      });
      
      console.log(`\n\n${'='.repeat(120)}`);
      console.log('RECOMMENDED DELETIONS');
      console.log('='.repeat(120));
      
      const toDelete = [];
      duplicateGroups.forEach(group => {
        // Keep the one with the most complete address
        const sorted = group.sort((a, b) => b.address.length - a.address.length);
        const keep = sorted[0];
        const remove = sorted.slice(1);
        
        console.log(`\nKeep: ID ${keep.id} - ${keep.title} (${keep.address})`);
        remove.forEach(job => {
          console.log(`  DELETE: ID ${job.id} - ${job.title} (${job.address})`);
          toDelete.push(job.id);
        });
      });
      
      console.log(`\n\nTotal jobs to delete: ${toDelete.length}`);
      console.log(`Jobs after deletion: ${rows.length - toDelete.length}`);
      console.log(`\nDELETE FROM jobPins WHERE id IN (${toDelete.join(', ')});`);
      
    } else {
      console.log('\nNo duplicate GPS coordinates found!');
    }
    
  } finally {
    await connection.end();
  }
}

analyzeDuplicates().catch(console.error);
