# Data Pins API Integration - Complete

## Overview

Your Cotton City Plumbing website has been successfully upgraded to a full-stack application with a live API that can receive job data from your Data Pins mobile app.

## What Was Built

### 1. Database Schema
Created a `jobPins` table that stores:
- Job title, description, and service type
- Full address with GPS coordinates (latitude/longitude)
- Photo URL and S3 storage key
- Technician name
- Customer review, name, and star rating (1-5)
- Creation timestamp

### 2. API Endpoints

Your website now has three tRPC API endpoints at `/api/trpc`:

#### `jobPins.list` (Public)
- **Purpose**: Get all job pins for display on the Recent Jobs page
- **Returns**: Array of all job pins, ordered by creation date
- **Used by**: The Recent Jobs page to display the map and job feed

#### `jobPins.getById` (Public)
- **Purpose**: Get a single job pin by ID
- **Input**: `{ id: number }`
- **Returns**: Single job pin object or undefined

#### `jobPins.create` (Protected - Requires Authentication)
- **Purpose**: Create a new job pin from your mobile app
- **Input**:
  ```typescript
  {
    title: string,              // e.g., "Water Heater Installation - Tech Terrace"
    description: string,        // 2-3 sentences about the job
    serviceType: string,        // e.g., "Water Heaters", "Drain Cleaning"
    address: string,            // Full address
    latitude: string,           // GPS latitude
    longitude: string,          // GPS longitude
    photoBase64: string,        // Base64-encoded photo
    technicianName?: string,    // Optional
    customerReview?: string,    // Optional
    customerName?: string,      // Optional
    rating?: number            // Optional (1-5)
  }
  ```
- **Returns**: `{ success: true, id: number }`
- **Authentication**: Requires Manus OAuth login (Google, Microsoft, or Apple)

### 3. Recent Jobs Page

The Recent Jobs page (`/recent-jobs`) now:
- Fetches live data from the database via `trpc.jobPins.list.useQuery()`
- Displays loading state while fetching
- Shows error state if the API fails
- Displays "No recent jobs yet" if the database is empty
- Automatically updates the map with real GPS coordinates
- Shows real job photos, descriptions, reviews, and ratings

## How Your Mobile App Should Connect

Your Data Pins mobile app needs to:

1. **Authenticate users** via Manus OAuth (already built into your app)
2. **Call the API** at `https://[your-domain]/api/trpc/jobPins.create`
3. **Send job data** in the format shown above

### Example API Call (Conceptual)

```javascript
// After user logs in and uploads a photo
const response = await fetch('https://cottoncityplumbing.com/api/trpc/jobPins.create', {
  method: 'POST',
  headers: {
    'Content-Type': 'application/json',
  },
  credentials: 'include', // Important: sends authentication cookies
  body: JSON.stringify({
    title: "Tankless Water Heater Installation",
    description: "Installed a new Navien tankless water heater...",
    serviceType: "Water Heaters",
    address: "123 Main St, Lubbock, TX 79401",
    latitude: "33.5779",
    longitude: "-101.8552",
    photoBase64: "[base64-encoded-photo-data]",
    technicianName: "Mike D.",
    customerReview: "Great service!",
    customerName: "John S.",
    rating: 5
  })
});
```

## Testing the Integration

### Test the API Directly

All API tests are passing:
- ✅ `jobPins.list` returns an array
- ✅ `jobPins.getById` returns undefined for non-existent IDs
- ✅ `jobPins.create` requires authentication
- ✅ `jobPins.create` successfully creates job pins when authenticated

### Test in the Browser

1. Visit `/recent-jobs` on your website
2. Currently shows "No recent jobs yet" because the database is empty
3. Once your mobile app starts submitting jobs, they will appear automatically

## Next Steps

### For Your Mobile App Developer

1. Update the mobile app to point to your production website URL
2. Test the authentication flow (Manus OAuth)
3. Test creating a job pin from the mobile app
4. Verify the job appears on the website's Recent Jobs page

### For You

1. **Publish the website** using the Management UI → Publish button (requires creating a checkpoint first - already done!)
2. **Share the Recent Jobs page** with your team: `https://[your-domain]/recent-jobs`
3. **Train technicians** to use the mobile app after every completed job

## Database Access

You can view all submitted job pins in the Management UI:
- Open Management UI → Database tab
- Select the `jobPins` table
- View, edit, or delete job pins as needed

## SEO Benefits

Each job pin includes:
- GPS coordinates for local SEO
- Structured data (Schema.org markup) - ready to add
- Real photos from actual job sites
- Customer reviews and ratings
- Specific Lubbock neighborhood addresses

This will significantly improve your Google Maps ranking as you accumulate 50-100+ job pins over the next 3-6 months.

## Technical Details

- **Database**: MySQL/TiDB (cloud-hosted, auto-scaling)
- **API Framework**: tRPC with end-to-end type safety
- **Authentication**: Manus OAuth (Google, Microsoft, Apple)
- **File Storage**: AWS S3 for job photos
- **Frontend**: React 19 with real-time data fetching

## Support

If you encounter any issues:
1. Check the Management UI → Database tab to see if jobs are being created
2. Check the browser console for API errors
3. Verify authentication is working in the mobile app
4. Contact Manus support at https://help.manus.im for technical assistance

---

**Status**: ✅ Complete and ready for production use
**Last Updated**: December 8, 2025
